import inspect
import logging
import time
import weakref
from functools import partial


logger = logging.getLogger(__name__)


class Tag(object):
    """``class`` that abstract tags and act as a simple container."""
    def __init__(self, name, valuable, ttl=-1, refresh=None, parent=None):
        """
        :param string name: tag name
        :param valuable: something that gives tag value, string or callable
        :param None,int ttl: Time to live for caching the tags, possible values
            are:
                * None (means no ttl)
                * -1 (means infinite ttl)
                * positive integer in seconds

            Note: this has absolutely not any importance for the cc-node, only
                for cc-server
        :param None,int refresh: period used to refresh tag value on the node
        :param object obj: parent object the tag is attached to, it may be
            needed for the tag callable to process the tag
        """
        self.name = name
        self.value = None
        self.is_function = False
        self.ttl = ttl
        self.refresh = refresh
        self.parent = parent if parent is None else weakref.proxy(parent)


        if inspect.isfunction(valuable):
            self.is_function = True
            args_count = len(inspect.getargspec(valuable).args)
            if args_count > 1:
                raise TypeError('Tag function take at most 1 argument')
            elif args_count == 1:
                self._calculate_value = partial(valuable, self.parent)
            else:
                self._calculate_value = valuable
        else:
            self.value = valuable

        self.watcher = None

    def calculate_value(self):
        self.value = self._calculate_value()
        # logger.debug('Calculate Tag(%s) = %s', self.name, self.value)

    def start(self, loop):
        """
        :param loop: pyev loop
        """
        if not self.is_function:
            return

        if self.refresh is None:
            self.calculate_value()
            return

        # TODO more sophisticated calculation with event propagation
        self.watcher = loop.timer(.0, float(self.refresh), lambda *args:
                                  self.calculate_value())
        self.watcher.start()

    def stop(self):
        if watcher is not None:
            self.watcher.stop()


def tag_inspector(mod, parent=None):
    """Inspect module to find tags.

    :param module mod: module to inspect

    Currently there are two ways to define a tag inside a module:

        * affect a string to a variable, the variable name will be the tag
          name
        * define a function that returns a value as a string or None (meaning
          the tag doesn't exist on the host), as you guessed the function name
          will define the tag name
    """
    tags = []
    for n, m in inspect.getmembers(mod):  # (name, member)
        # only keep strings or functions as tags
        if getattr(m, '__module__', None) != mod.__name__ or (
            n.startswith('_')):
            continue
        elif isinstance(m, (str, unicode)):
            # if string, it means it is constant, then ttl = -1
            ttl = -1
            refresh = None
        elif inspect.isfunction(m):
            # if function take function ttl argument or set -1 by default
            ttl = getattr(m, 'ttl', -1)
            refresh = getattr(m, 'refresh', None)
        else:
            # whatever it is we don't care...
            continue

        logger.debug('Introspected %s with ttl %s.', n, ttl)

        # finally add the tag
        tags.append(Tag(n, m, ttl, refresh, parent))

    return tags


# decorators for tag inspector
def ttl(value):
    def decorator(func):
        func.ttl = value
        return func
    return decorator


def refresh(value):
    def decorator(func):
        func.refresh = value
        return func
    return decorator


def get_tags(tags_dict, tags=None, noresolve_tags=None):
    """Helper to get tags.

    :param tags_dict: dict containing :class:`Tag` objects
    :param tags: list of tags to get (None mean all tags)
    :param noresolve_tags: list of tags to exclude from result
    """
    logger.debug('Tags request: %s, %s', unicode(tags), unicode(noresolve_tags))

    tags = set(tags) - set(noresolve_tags) if tags is not None else None

    if tags is None:
        tags = tags_dict.iterkeys()
    else:
        tags = tags & set(tags_dict)

    profile = time.time()
    result = dict((
        t,  # tag name
        dict(
            value=tags_dict[t].value,
            ttl=tags_dict[t].ttl,
        ),
    ) for t in tags)

    logger.debug('Profiling: %f seconds', time.time() - profile)
    logger.debug('Returning: %s', unicode(result))
    return result
