#!/usr/bin/env python
#coding=utf8

'''
Local client representation classes.
'''

from threading import Lock
from datetime import datetime

from sjrpc.utils import ConnectionProxy

class CCClient(object):
    '''
    Represent a single client connected to the server.
    '''

    def __init__(self, login, role, server, connection):

        # The login of the client:
        self.login = login

        # The role of the client:
        self.role = role

        # The server binded to this client:
        self.server = server

        # The connection of the client (public attribute):
        self.connection = connection

        # The date of connection of the client:
        self._connection_date = datetime.now()

        # The date of the last action:
        self._last_action = datetime.now()

        # The connection proxy:
        self.proxy = ConnectionProxy(connection)

        # Jobs lock for this client:
        self.lock = Lock()

    def get_uptime(self):
        '''
        Get the uptime of the client connection in seconds.

        :return: uptime of the client
        '''

        dt = datetime.now() - self._connection_date
        return dt.seconds + dt.days * 86400

    def get_idle(self):
        '''
        Get the idle time of the client connection in seconds.

        :return: idle of the client
        '''

        dt = datetime.now() - self._last_action
        return dt.seconds + dt.days * 86400

    def top(self):
        '''
        Reset the last_action date to now.
        '''

        self._last_action = datetime.now()

    def get_ip(self):
        peer = self.connection.getpeername()
        return ':'.join(peer.split(':')[:-1])

    def shutdown(self):
        '''
        Shutdown the connection to the client.
        '''

        self.server.rpc.unregister(self.connection, shutdown=True)

    def get_tags(self):
        '''
        Get all server defined tags.
        '''

        tags = {}
        tags['con'] = self.get_uptime()
        tags['ip'] = self.get_ip()
        return tags

    def get_remote_tags(self, tag):
        return self.connection.call('get_tags', (tag,))[tag]
