'''
Main class of cc-server.
'''

from __future__ import absolute_import

import logging
from fnmatch import fnmatch as glob
from functools import partial

from sjrpc.server import SSLRpcServer

from ccserver.handlers import WelcomeHandler
from ccserver.conf import CCConf
from ccserver.client import CCClient
from ccserver.exceptions import AlreadyRegistered, NotConnectedAccountError
from ccserver.jobs import JobsManager

from cloudcontrol.common.tql.db.object import TqlObject
from cloudcontrol.common.tql.db.tag import StaticTag, CallbackTag
from cloudcontrol.common.tql.db.db import TqlDatabase


class CCServer(object):
    '''
    CloudControl server main class.

    :param conf_dir: the directory that store the client configuration
    :param certfile: the path to the ssl certificate
    :param keyfile: the path to the ssl key
    :param address: the interface to bind
    :param port: the port to bind
    '''

    LISTEN_BACKLOG = 5

    # These tags are reserved and cannot be setted by an user:
    RESERVED_TAGS = ('id', 'a', 'r', 'close', 'con', 'ip', 'p')

    def __init__(self, conf_dir, maxcon, maxidle, certfile=None, keyfile=None,
                 address='0.0.0.0', port=1984):

        # Dict containing all connected accounts, the key is the login of
        # the account and the value the :class:`RpcConnection` of the peer:
        self._connected = {}
        # The interface object to the configuration directory:
        self.conf = CCConf(conf_dir)
        # Some settings:
        self._maxcon = maxcon
        self._maxidle = maxidle
        # SSL configuration stuff:
        if certfile:
            logging.info('SSL Certificate: %s', certfile)
        if keyfile:
            logging.info('SSL Key: %s', certfile)

        self.db = TqlDatabase()

        # Create the rpc server:
        logging.info('Listening on %s:%s', address, port)
        self.rpc = SSLRpcServer.from_addr(address, port, certfile=certfile,
                                          keyfile=keyfile,
                                          conn_kw=dict(handler=WelcomeHandler(self),
                                                       on_disconnect='on_disconnect'))

        # The jobs manager:
        self.jobs = JobsManager(self)
        logging.info('Server started to running')

    def _update_accounts(self):
        '''
        Update the database with accounts.
        '''

        db_accounts = set((obj['a'].get_value() for obj in self.db.objects if 'a' in obj))
        accounts = set(self.conf.list_accounts())

        to_register = accounts - db_accounts
        to_unregister = db_accounts - accounts

        for login in to_register:
            conf = self.conf.show(login)
            obj = TqlObject(login)
            obj.register(StaticTag('r', conf['role']))
            obj.register(StaticTag('a', login))
            self.db.register(obj)

        for login in to_unregister:
            self.db.unregister(login)

    def iter_connected_role(self, role=None):
        '''
        Generator to iter on each connected client with specified role. If role
        is None, return all connected clients.

        :param role: role to filter
        '''

        for login, client in self._connected.items():
            if role is None or client.role == role:
                yield client

    def register(self, login, role, connection, create_object=False):
        '''
        Register a new connected account on the server.

        :param login: login of the account
        :param connection: connection to register
        :param tags: tags to add for the client
        :param create_object: create the object on objectdb
        '''

        if login in self._connected:
            raise AlreadyRegistered('A client is already connected with this '
                                    'account.')
        else:
            self._connected[login] = CCClient(login, role, self, connection)

        # Create the object on objectdb if required:
        if create_object:
            obj = TqlObject(login)
            obj.register(StaticTag('r', role))
            self.db.register(obj)
        else:
            obj = self.db.get(login)
            assert obj is not None

        # Define server defined tags for the new node:
        obj.register(CallbackTag('con', self._connected[login].get_uptime, ttl=0))
        obj.register(CallbackTag('ip', self._connected[login].get_ip))

    def unregister(self, connection):
        '''
        Unregister an already connected account.

        :param connection: the connection of the client to unregister
        '''
        #XXX: not functional since new db!!!
        client = self.search_client_by_connection(connection)

        # Unregister objects from database if it have no account attached:
        obj = self.db.get(client.login)
        if obj is not None and 'a' not in obj:
            self.db.unregister(obj)
        else:
            # Unregister tags of connected account:
            obj.unregister()

        if client.login in self._connected:
            del self._connected[client.login]
        #self.objects.unregister_children(client.login)

    def sub_register(self, parent, name, role):
        '''
        Register a new node supervised by a parent.

        :param parent: the parent login of the subnode
        :param login: the name of the subnode
        :param role: the role of the subnode
        '''

        client = self.get_connection(parent)
        child = '%s.%s' % (parent, name)
        client.register_child(child)

        # Register the children in the tags database:
        obj = TqlObject(child)
        obj.register(StaticTag('r', role))
        obj.register(StaticTag('p', client))
        self.db.register(obj)

    def sub_unregister(self, parent, name):
        '''
        Unregister a node supervised by a parent.

        :param parent: the parent of the subnode
        :param login: the name of the subnode
        '''

        client = self.get_connection(parent)
        child = '%s.%s' % (parent, name)
        client.unregister_child(child)

        # Unregister the children from the tags database:
        self.objects.unregister(child)

    def search_client_by_connection(self, connection):
        '''
        Search a connected client by it connection. If no client is found,
        return None.

        :param connection: the connection of the client to search
        :return: the found client or None
        '''

        for client in self._connected.values():
            if client.connection is connection:
                return client
        else:
            return None

    def run(self):
        '''
        Run the server mainloop.
        '''

        # Register accounts on the database:
        self._update_accounts()

        # Running server internal jobs:
        self.jobs.create('kill_oldcli', maxcon=self._maxcon,
                         maxidle=self._maxidle, _hidden=True)

        logging.debug('Running rpc mainloop')
        self.rpc.run()

    def get_connection(self, login):
        '''
        Get the connection of a connecter account login.

        :param login: login of the connection to get
        :return: :class:`RpcConnection` instance of the peer connection
        '''

        return self._connected[login]

    def kill(self, login):
        '''
        Disconnect from the server the client identified by provided login.

        :param login: the login of the user to disconnect
        :throws NotConnectedAccount: when provided account is not connected (or
            if account doesn't exists).
        '''

        client = self._connected.get(login)
        if client is None:
            raise NotConnectedAccountError('The account %s is not '
                                           'connected' % login)
        client.shutdown()

    def check(self, login, method, tql=None):
        '''
        Check if the user can call the specified method with specified TQL.

        :param login: the login of the user
        :param method: the method to call
        :param tql: the tql passed in argument of the method
        :return: True if user have rights, else False
        '''

        rights = self.conf.show(login)['rights']
        if tql is not None:
            objects = self.db.raw_query(tql)
        for right in rights:
            if not (right['method'] is None or glob(method, right['method'])):
                continue
            if tql is not None and right['tql']:
                objects_right = self.db.raw_query(right['tql'])
                if set(objects) <= set(objects_right):
                    return right['target'] == 'allow'
            else:
                return right['target'] == 'allow'
        return False

    def tags_register(self, login, name, ttl=None, value=None):
        '''
        Register a new tag for a client.

        :param login: login of the client
        :param name: name of the tag to register
        :param ttl: TTL of the tag if applicable (None = no TTL, the tag will
                never expire)
        :param value: value of the tag
        '''

        obj = self.db.get(login)
        client = self._connected.get(login)
        callback = partial(client.get_remote_tags, name)
        tag = CallbackTag(name, callback, ttl=ttl)
        obj.register(tag)

    def tags_unregister(self, login, name):
        '''
        Unregister a tag for the client.

        :param login: login of the client
        :param name: name of the tag to unregister
        '''

        obj = self.db.get(login)
        obj.unregister(name)

    def tags_drop(self, login, name):
        '''
        Drop the cached value of a tag for the client.

        :param login: login of the client
        :param name: name of the tag to drop
        '''
        obj = self.db.get(login)
        tag = obj.get(name)
        if tag is not None:
            tag.invalidate()

    def tags_update(self, login, name, value=None, ttl=None):
        '''
        Update a tag.

        :param login: login of the client
        :param name: name of the tag to update
        :param value: new value of the tag
        :param ttl: new ttl of the tag
        '''
        obj = self.db.get(login)
        tag = obj.get(name)
        if tag is not None:
            if value is not None:
                tag.set_value(value)
            if ttl is not None:
                tag.ttl = ttl

    def list(self, query, show=None):
        self._update_accounts()
        return self.db.query(query, show)
