'''
Main class of cc-server.
'''

from __future__ import absolute_import

import logging
from fnmatch import fnmatch as glob

from sjrpc.server import SSLRpcServer
from sjrpc.utils import RpcHandler, pass_connection

from ccserver.conf import CCConf
from ccserver.exceptions import AlreadyRegistered, NotConnectedAccountError, AuthenticationError, BadRoleError
from ccserver.jobs import JobsManager
from ccserver.clients import Client

#from cloudcontrol.common.tql.db.object import TqlObject
from ccserver.db import SObject
from cloudcontrol.common.tql.db.tag import StaticTag
from cloudcontrol.common.tql.db.db import TqlDatabase

# Import all enabled roles
import ccserver.clients.cli
import ccserver.clients.host
import ccserver.clients.hv
import ccserver.clients.bootstrap
import ccserver.clients.spv


class WelcomeHandler(RpcHandler):
    """ Default handler used on client connections of the server.
    """

    def __init__(self, server):
        self._server = server

    @pass_connection
    def authentify(self, conn, login, password):
        return self._server.authenticate(conn, login, password)


class CCServer(object):
    '''
    CloudControl server main class.

    :param conf_dir: the directory that store the client configuration
    :param certfile: the path to the ssl certificate
    :param keyfile: the path to the ssl key
    :param address: the interface to bind
    :param port: the port to bind
    '''

    LISTEN_BACKLOG = 50

    # These tags are reserved and cannot be setted by an user:
    RESERVED_TAGS = ('id', 'a', 'r', 'close', 'con', 'ip', 'p')

    def __init__(self, conf_dir, maxcon, maxidle, certfile=None, keyfile=None,
                 address='0.0.0.0', port=1984):

        self._clients = {}  # Clients connected to the server

        # The interface object to the configuration directory:
        self.conf = CCConf(conf_dir)
        # Some settings:
        self._maxcon = maxcon
        self._maxidle = maxidle
        # SSL configuration stuff:
        if certfile:
            logging.info('SSL Certificate: %s', certfile)
        if keyfile:
            logging.info('SSL Key: %s', certfile)

        self.db = TqlDatabase()

        # Create the rpc server:
        logging.info('Listening on %s:%s', address, port)
        self.rpc = SSLRpcServer.from_addr(address, port, certfile=certfile,
                                          keyfile=keyfile,
                                          conn_kw=dict(handler=WelcomeHandler(self),
                                                       on_disconnect='on_disconnect'))

        # The jobs manager:
        self.jobs = JobsManager(self)
        logging.info('Server started to running')

    def _update_accounts(self):
        '''
        Update the database with accounts.
        '''

        db_accounts = set((obj['a'].get_value() for obj in self.db.objects if 'a' in obj))
        accounts = set(self.conf.list_accounts())

        to_register = accounts - db_accounts
        to_unregister = db_accounts - accounts

        for login in to_register:
            conf = self.conf.show(login)
            obj = SObject(login)
            obj.register(StaticTag('r', conf['role']), override=True)
            obj.register(StaticTag('a', login), override=True)
            # Register static tags:
            for tag, value in self.conf.show(login)['tags'].iteritems():
                obj.register(StaticTag(tag, value), override=True)
            self.db.register(obj)

        for login in to_unregister:
            self.db.unregister(login)

    def iter_connected_role(self, role=None):
        '''
        Generator to iter on each connected client with specified role. If role
        is None, return all connected clients.

        :param role: role to filter
        '''

        for login, client in self._clients.items():
            if role is None or client.role == role:
                yield client

    def authenticate(self, conn, login, password):
        """ Authenticate a client against provided login and password.

        If the authentication is a success, register the client on the server
        and return the client role, else, raise an exception.
        """

        logmsg = 'Authentication error from %s: '
        with self.conf:
            try:
                role = self.conf.authentify(login, password)
            except CCConf.UnknownAccount:
                raise AuthenticationError('Unknown login')
            else:
                if 'close' in self.conf.show(login)['tags']:
                    logging.warning(logmsg + 'account closed (%s)', conn.getpeername(), login)
                    raise AuthenticationError('Account is closed')

        if role is None:
            logging.warning(logmsg + 'bad login/password (%s)', conn.getpeername(), login)
            raise AuthenticationError('Bad login/password')
        else:
            if role not in Client.roles:
                logging.warning(logmsg + 'bad role in account config (%s)', conn.getpeername(), login)
                raise BadRoleError('%r is not a legal role' % role)

            create_object = False

            # If authentication is a success, try to register the client:
            if role == 'bootstrap':
                # Set a bootstrap id for the object:
                login = '%s.%s' % (login, conn.get_fd())
                # Real role of the node will be host:
                role = 'host'
                create_object = True

            # Try to register the client:
            for _ in xrange(5):
                try:
                    self.register(login, role, conn, create_object)
                except AlreadyRegistered:
                    if role == 'cli':
                        try:
                            self.kill(login)
                        except NotConnectedAccountError:
                            pass
                else:
                    break
            else:
                logging.warning(logmsg + 'already connected (%s)', conn.getpeername(), login)
                raise AuthenticationError('Already connected')

            logging.info('Authentication success from %s with login %s', conn.getpeername(), login)
            return role


    def register(self, login, role, connection, create_object=False):
        '''
        Register a new connected account on the server.

        :param login: login of the account
        :param connection: connection to register
        :param tags: tags to add for the client
        :param create_object: create the object on objectdb
        '''

        # Create the object on objectdb if required:
        if create_object:
            tql_object = SObject(login)
            tql_object.register(StaticTag('r', role))
            self.db.register(tql_object)
        else:
            tql_object = self.db.get(login)
            assert tql_object is not None

        # Register the client:
        if login in self._clients:
            raise AlreadyRegistered('A client is already connected with this account.')
        else:
            client = Client.from_role(role, login, self, connection, tql_object)
            self._clients[login] = client

    def unregister(self, client):
        """ Unregister a client.
        """
        del self._clients[client.login]

    def run(self):
        '''
        Run the server mainloop.
        '''

        # Register accounts on the database:
        self._update_accounts()

        # Running server internal jobs:
        self.jobs.create('kill_oldcli', maxcon=self._maxcon,
                         maxidle=self._maxidle, _hidden=True)

        logging.debug('Running rpc mainloop')
        self.rpc.run()

    def get_client(self, login):
        """ Get a connected client by its login.

        :param login: login of the connection to get
        :return: the client instance
        """
        return self._clients[login]

    def kill(self, login):
        '''
        Disconnect from the server the client identified by provided login.

        :param login: the login of the user to disconnect
        :throws NotConnectedAccount: when provided account is not connected (or
            if account doesn't exists).
        '''

        client = self._clients.get(login)
        if client is None:
            raise NotConnectedAccountError('The account %s is not '
                                           'connected' % login)
        client.shutdown()

    def check(self, login, method, tql=None):
        '''
        Check if the user can call the specified method with specified TQL.

        :param login: the login of the user
        :param method: the method to call
        :param tql: the tql passed in argument of the method
        :return: True if user have rights, else False
        '''

        rights = self.conf.show(login)['rights']
        if tql is not None:
            objects = self.db.raw_query(tql)
        for right in rights:
            if not (right['method'] is None or glob(method, right['method'])):
                continue
            if tql is not None and right['tql']:
                objects_right = self.db.raw_query(right['tql'])
                if set(objects) <= set(objects_right):
                    return right['target'] == 'allow'
            else:
                return right['target'] == 'allow'
        return False

    def list(self, query, show=None):
        self._update_accounts()
        return self.db.query(query, show)
