#!/usr/bin/env python
#coding=utf8

from __future__ import absolute_import

import inspect
import logging

from sjrpc.utils import RpcHandler
from sjrpc.core import RpcError

from ccserver.orderedset import OrderedSet
from ccserver.conf import CCConf
from ccserver.exceptions import (AlreadyRegistered, AuthenticationError,
                                 RightError, ReservedTagError, BadObjectError,
                                 BadRoleError, NotConnectedAccountError,
                                 CloneError)

from ccserver import __version__


def listed(func):
    func.__listed__ = True
    return func


class Reporter(object):
    '''
    Simple class used to report error, warning and success of command execution.
    '''

    def __init__(self):
        self._reports = []

    def get_dict(self):
        return {'objects': self._reports,
                'order': ['id', 'status', 'message', 'output']}

    def success(self, oid, message, output=None):
        self._reports.append({'id': oid, 'status': 'success',
                              'message': message, 'output': output})

    def warn(self, oid, message, output=None):
        self._reports.append({'id': oid, 'status': 'warn',
                              'message': message, 'output': output})

    def error(self, oid, message, output=None):
        self._reports.append({'id': oid, 'status': 'error',
                              'message': message, 'output': output})


class CCHandler(RpcHandler):
    """ Base class for handlers of CloudControl server.

    This class provide the following features:

    - functions can be used to get all functions decorated with the listed
        decorator
    - version can be used to get the current cc-server version
    """

    def __init__(self, client):
        self.client = client
        self.server = client.server  # Shortcut to server
        self.conf = client.server.conf  # Shortcut to configuration

    def __getitem__(self, name):
        if name.startswith('_'):
            # Filter the private members access:
            raise KeyError('Remote name %s is private.' % repr(name))
        else:
            logging.debug('Called %s.%s', self.__class__.__name__, name)
            return super(CCHandler, self).__getitem__(name)

    @listed
    def functions(self):
        '''
        Show the list of functions available to the peer.

        :return: list of dict with keys name and description.
        '''

        cmd_list = []

        for attr in dir(self):
            attr = getattr(self, attr, None)
            if getattr(attr, '__listed__', False):
                cmd = {}
                cmd['name'] = attr.__name__
                doc = inspect.getdoc(attr)
                if doc:
                    cmd['description'] = inspect.cleandoc(doc)
                cmd_list.append(cmd)

        return cmd_list

    @listed
    def version(self):
        '''
        Return the current server version.

        :return: the version
        '''
        return __version__
