# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


""" Connected client management package.

This package store classes representing each client's role and the associated
sjRPC handler.
"""

from functools import partial
from datetime import datetime

from sjrpc.utils import ConnectionProxy, threadless
from sjrpc.core import RpcError

from cloudcontrol.server.handlers import CCHandler, listed
from cloudcontrol.server.exceptions import RightError
from cloudcontrol.server.db import RemoteTag

from cloudcontrol.common.tql.db.tag import StaticTag, CallbackTag
from cloudcontrol.common.tql.db.object import TqlObject


class CCServerConnectionProxy(ConnectionProxy):

    """ RPC proxy used to add the node login in the exception message.
    """

    def __init__(self, login, *args, **kwargs):
        super(CCServerConnectionProxy, self).__init__(*args, **kwargs)
        self._login = login

    def __getattr__(self, name):
        def func(*args, **kwargs):
            try:
                return super(CCServerConnectionProxy, self).__getattr__(name)(*args, **kwargs)
            except RpcError as err:
                err.message = '%s: %s' % (self._login, err.message)
                raise err
            except Exception as err:
                raise err.__class__('%s: %s' % (self._login, str(err)))
        return func


class RegisteredCCHandler(CCHandler):

    """ Basic handler for all registered clients.
    """

    def __getitem__(self, name):
        self.client.top()
        return super(RegisteredCCHandler, self).__getitem__(name)

    def on_disconnect(self, conn):
        self.logger.info('Client %s disconnected', self.client.login)
        self.client.shutdown()

    #
    # Tags registration handler functions:
    #

    @threadless
    def tags_register(self, name, ttl=None, value=None):
        """ Register a new tag on the calling node.

        :param name: name of the tag to register
        :param ttl: ttl of the tag (or None if not applicable)
        :param value: value to fill the tag (optionnal)
        """
        self.client.tags_register(name, ttl, value)

    @threadless
    def tags_unregister(self, name):
        """ Unregister a tag on the calling node.

        :param name: name of the tag to unregister
        """
        self.client.tags_unregister(name)

    def tags_drop(self, name):
        """ Drop the tag value of the specified tag on the calling node.

        :param name: name of the tag to drop
        """
        self.client.tags_drop(name)

    def tags_update(self, name, value, ttl=None):
        """ Update the value of the specified tag on the calling node.

        :param name: name of the tag to update
        :param value: new tag value
        :param ttl: new ttl value
        """
        self.client.tags_update(name, value, ttl)

    #
    # Sub objects functions:
    #

    @listed
    @threadless
    def register(self, obj_id, role):
        """ Register an object managed by the calling node.

        .. note:
           the obj_id argument passed to this handler is the object id of the
           registered object (not the fully qualified id, the server will
           preprend the id by "node_id." itself).

        :param obj_id: the id of the object to register
        :param role: the role of the object to register
        """
        self.client.register(obj_id, role)

    @listed
    @threadless
    def unregister(self, obj_id):
        """ Unregister an object managed by the calling node.

        .. note:
           the obj_id argument passed to this handler is the object id of the
           unregistered object (not the fully qualified id, the server will
           preprend the id by "node_id." itself).

        :param obj_id: the id of the object to unregister
        """
        self.client.unregister(obj_id)

    @listed
    @threadless
    def sub_tags_register(self, obj_id, name, ttl=None, value=None):
        """ Register a new remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to register
        :param ttl: TTL of the tag if applicable (None = no TTL, the tag will
                never expire)
        :param value: value of the tag
        """
        self.client.sub_tags_register(obj_id, name, ttl, value)

    @listed
    @threadless
    def sub_tags_unregister(self, obj_id, name):
        """ Unregister a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to unregister
        """
        self.client.sub_tags_unregister(obj_id, name)

    @listed
    def sub_tags_drop(self, obj_id, name):
        """ Drop the cached value of a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to drop
        """
        self.client.sub_tags_drop(obj_id, name)

    @listed
    def sub_tags_update(self, obj_id, name, value=None, ttl=None):
        """ Update a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to update
        :param value: new value of the tag
        :param ttl: new ttl of the tag
        """
        self.client.sub_tags_update(obj_id, name, value, ttl)


class Client(object):

    """ Base class for all types cc-server clients.

    :param login: login of the client
    :param server: server instance
    :param connection: rpc connection to the client
    """

    ROLE = None
    RPC_HANDLER = RegisteredCCHandler
    KILL_ALREADY_CONNECTED = False

    roles = {}

    def __init__(self, logger, login, server, connection):
        self.logger = logger
        self._login = login
        self._server = server
        self._connection = connection
        self._handler = self.RPC_HANDLER(self)
        self._proxy = CCServerConnectionProxy(login, self._connection.rpc)
        self._last_action = datetime.now()
        self._connection_date = datetime.now()
        self._tql_object = None
        self._children = {}
        self._server.conf.set_last_connection(self._login)

        # Remote tags registered:
        self._remote_tags = set()

    def _get_tql_object(self):
        """ Get the TQL object of the client from the cc-server tql database.
        """
        return self._server.db.get(self.login)

    @classmethod
    def register_client_class(cls, class_):
        """ Register a new client class.
        """
        cls.roles[class_.ROLE] = class_

    @classmethod
    def from_role(cls, role, logger, login, server, connection):
        return cls.roles[role](logger, login, server, connection)

    #
    # Properties
    #

    @property
    def proxy(self):
        """ Return a proxy to the rpc.
        """
        return self._proxy

    @property
    def object(self):
        """ Return the tql object of this client.
        """
        return self._tql_object

    @property
    def account(self):
        """ Return the login (account) of this client.
        """
        return self._login

    @property
    def login(self):
        """ Return the login of this client.
        """
        return self._login

    @property
    def role(self):
        """ Return the role of this client.
        """
        return self.ROLE

    @property
    def server(self):
        """ Return the cc-server binded to this client.
        """
        return self._server

    @property
    def conn(self):
        """ Return the sjrpc connection to the client.
        """
        return self._connection

    @property
    def uptime(self):
        """ Get the uptime of the client connection in seconds.

        :return: uptime of the client
        """

        dt = datetime.now() - self._connection_date
        return dt.seconds + dt.days * 86400

    @property
    def idle(self):
        """ Get the idle time of the client connection in seconds.

        :return: idle of the client
        """
        dt = datetime.now() - self._last_action
        return dt.seconds + dt.days * 86400

    @property
    def ip(self):
        """ Get client remote ip address.
        """
        peer = self.conn.getpeername()
        return ':'.join(peer.split(':')[:-1])

    def attach(self):
        """ Attach the client to the server.
        """

        # Set the role's handler for the client:
        self.conn.rpc.set_handler(self._handler)

        # Register the client in tql database:
        self._tql_object = self._get_tql_object()

        # Register the server defined client tags:
        self._tql_object.register(CallbackTag('con', lambda: self.uptime, ttl=0))
        self._tql_object.register(CallbackTag('idle', lambda: self.idle, ttl=0))
        self._tql_object.register(CallbackTag('ip', lambda: self.ip))

    def shutdown(self):
        """ Shutdown the connection to the client.
        """

        self._server.conf.set_last_disconnection(self._login)

        # Unregister all children:
        for child in self._children.copy():
            self.unregister(child)

        # Disable the client handler:
        self.conn.rpc.set_handler(None)

        # Unregister all remote tags:
        for tag in self._remote_tags.copy():
            self.tags_unregister(tag)

        # Unrefister all server defined client tags:
        self._tql_object.unregister('con')
        self._tql_object.unregister('idle')
        self._tql_object.unregister('ip')

        self._server.rpc.unregister(self.conn, shutdown=True)
        self._server.unregister(self)

    def top(self):
        """ Reset the "last action" date to now.
        """
        self._last_action = datetime.now()

    def async_remote_tags(self, watcher, robj, tags):
        """ Asynchronously update tags from the remote client using
            specified watcher.
        """
        watcher.register(self.conn, 'get_tags', tags, _data=(tags, robj))

    def tags_register(self, name, ttl=None, value=None):
        """ Register a new remote tag for the client.

        :param name: name of the tag to register
        :param ttl: TTL of the tag if applicable (None = no TTL, the tag will
                never expire)
        :param value: value of the tag
        """

        tag = RemoteTag(name, self.async_remote_tags, ttl=ttl)
        self._tql_object.register(tag)
        self._remote_tags.add(name)

    def tags_unregister(self, name):
        """
        Unregister a remote tag for the client.

        :param name: name of the tag to unregister
        """

        self._tql_object.unregister(name)
        self._remote_tags.discard(name)

    def tags_drop(self, name):
        """ Drop the cached value of a remote tag for the client.

        :param name: name of the tag to drop
        """
        tag = self._tql_object.get(name)
        if tag is not None:
            tag.invalidate()

    def tags_update(self, name, value=None, ttl=None):
        """ Update a remote tag.

        :param name: name of the tag to update
        :param value: new value of the tag
        :param ttl: new ttl of the tag
        """
        tag = self._tql_object.get(name)
        if tag is not None:
            if value is not None:
                tag.cached = value
            if ttl is not None:
                tag.ttl = ttl

    def list(self, query, show=None, method=None):
        return self._server.list(query, show=show, method=method, requester=self.account)

    def check(self, method, query=None):
        if query is not None:
            if not self._server.check(query, self.account, method):
                raise RightError('You don\'t have right to do that')
        else:
            if not self._server.check_method(self.account, method):
                raise RightError('You don\'t have right to do that')

    def register(self, obj_id, role):
        """ Register a new child.
        """
        child = '%s.%s' % (self.login, obj_id)

        # Register the children in the tags database:
        obj = TqlObject(child)
        obj.register(StaticTag('r', role))
        obj.register(StaticTag('p', self.login))
        self._server.db.register(obj)
        self._children[obj_id] = obj

    def unregister(self, obj_id):
        """ Unregister a child.
        """
        child = '%s.%s' % (self.login, obj_id)
        del self._children[obj_id]
        # Unregister the children from the tags database:
        self._server.db.unregister(child)

    def sub_tags_register(self, obj_id, name, ttl=None, value=None):
        """ Register a new remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to register
        :param ttl: TTL of the tag if applicable (None = no TTL, the tag will
                never expire)
        :param value: value of the tag
        """

        callback = partial(self.async_remote_sub_tags, obj_id)
        tag = RemoteTag(name, callback, ttl=ttl)
        self._children[obj_id].register(tag)

    def sub_tags_unregister(self, obj_id, name):
        """ Unregister a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to unregister
        """

        self._children[obj_id].unregister(name)

    def sub_tags_drop(self, obj_id, name):
        """ Drop the cached value of a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to drop
        """
        tag = self._children[obj_id].get(name)
        if tag is not None:
            tag.invalidate()

    def sub_tags_update(self, obj_id, name, value=None, ttl=None):
        """ Update a remote tag for a child of the client.

        :param obj_id: child name
        :param name: name of the tag to update
        :param value: new value of the tag
        :param ttl: new ttl of the tag
        """
        tag = self._children[obj_id].get(name)
        if tag is not None:
            if value is not None:
                tag.cached = value
            if ttl is not None:
                tag.ttl = ttl

    def async_remote_sub_tags(self, obj_id, watcher, robj, tags):
        """ Asynchronously update sub tags from the remote client using
            specified watcher.
        """
        watcher.register(self.conn, 'sub_tags', obj_id, tags, _data=(tags, robj))