# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


from cloudcontrol.server.clients import Client, RegisteredCCHandler


class HostHandler(RegisteredCCHandler):

    """ Handler binded to an host client.
    """

    def script_get(self, name):
        """ Get a script by its name.

        :param name: name of the script to get
        """
        return self.server.scripts.load(name)

    def plugin_get(self, name):
        """ Get a plugin by its name.

        :param name: name of the plugin to get
        """
        return self.server.plugins.load(name)


class HostClient(Client):

    """ A host client connected to the cc-server.
    """

    ROLE = 'host'
    RPC_HANDLER = HostHandler

    def __init__(self, *args, **kwargs):
        super(HostClient, self).__init__(*args, **kwargs)

    def execute(self, command):
        return self.conn.call('execute_command', command)

    def shutdown_node(self, reboot=False, gracefull=True):
        """ Shutdown the remote node.
        """
        return self.conn.call('node_shutdown', reboot, gracefull)

    def console(self, name):
        """ Start a remote console on the specified vm.
        """
        label = self.proxy.vm_open_console(name)
        tun = self.conn.create_tunnel(label=label)
        return tun

    def shell(self):
        """ Start a remote shell on the host.
        """
        label = self.proxy.shell()
        tun = self.conn.create_tunnel(label=label)
        return tun

    def resize(self, label, *args, **kwargs):
        return self.proxy.resize(label, *args, **kwargs)

    def forward(self, port, destination='127.0.0.1'):
        """ Create a forwarding tunnel on this client and return it.
        """
        tun = self.conn.create_tunnel()
        self.proxy.forward(tun.label, port, destination)
        return tun

    def script_run(self, sha1_hash, script, owner, batch=None, *args):
        """ Run a script on the host.
        """
        return self.conn.call('script_run', sha1_hash, script, owner, batch, *args)

    def plugin_help(self, plugin, method):
        """ Get help about a plugin installed on matching host.
        """
        return self.conn.call('plugin_help', plugin, method)

    def plugin_run(self, plugin, method, owner, batch=None, **kwargs):
        """ Run a plugin method on the host.
        """
        return self.conn.call('plugin_run', plugin, method, owner,
                              batch=batch, **kwargs)

    def plugin_install(self, sha1_hash, plugin):
        """ Install a plugin on the host.
        """
        return self.conn.call('plugin_install', sha1_hash, plugin)

    def plugin_uninstall(self, plugin):
        """ Uninstall a plugin on the host.
        """
        return self.conn.call('plugin_uninstall', plugin)

    def job_wait(self, job_id):
        """ Wait until a job is done and return its output.
        """
        while True:
            job = self._server.db.get_by_id('.'.join((self.login, job_id)))
            if job is not None:
                if job['state'] == 'done':
                    return job


Client.register_client_class(HostClient)
