# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


import threading

from cloudcontrol.server.clients import Client
from cloudcontrol.server.clients.host import HostClient


class HypervisorHandler(HostClient.RPC_HANDLER):
    """ Handler binded to an hv client.
    """


class HvClient(HostClient):

    """ A hv client connected to the cc-server.
    """

    ROLE = 'hv'
    RPC_HANDLER = HypervisorHandler
    DEFINE_TIMEOUT = 60 * 10  # 10mn

    def __init__(self, *args, **kwargs):
        super(HvClient, self).__init__(*args, **kwargs)
        self._hv_lock = threading.RLock()

    @property
    def hvlock(self):
        """ The lock used on hypervirsors for migration.
        """
        return self._hv_lock

    #
    # Children specific methods:
    #

    def execute(self, command):
        return self.conn.call('execute_command', command)

    def get_child_remote_tags(self, obj_id, tag):
        return self.conn.call('sub_tags', obj_id, (tag,))[tag]

    def vm_action(self, action, vms, *args, **kwargs):
        return self.conn.call(action, vms, *args, **kwargs)

    def define(self, data, format):
        return self.conn.call('vm_define', data, format, _timeout=self.DEFINE_TIMEOUT)


Client.register_client_class(HvClient)
