# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


import os
import inspect
import time
import functools
import base64

from sjrpc.utils import RpcHandler

from cloudcontrol.server import __version__


def listed(func):
    func.__listed__ = True
    return func


class Reporter(object):
    """ Simple class used to report error, warning and success of command execution.
    """

    def __init__(self):
        self._reports = []

    def get_dict(self):
        return {'objects': self._reports,
                'order': ['id', 'status', 'message', 'output']}

    def success(self, oid, message, output=None, jobs=None):
        report = {'id': oid, 'status': 'success',
                  'message': message, 'output': output}
        if jobs is not None:
            if isinstance(jobs, basestring):
                jobs = [jobs]
            jobs = ' '.join(jobs)
            report['jobs'] = jobs
        self._reports.append(report)

    def warn(self, oid, message, output=None):
        self._reports.append({'id': oid, 'status': 'warn',
                              'message': message, 'output': output})

    def error(self, oid, message, output=None):
        self._reports.append({'id': oid, 'status': 'error',
                              'message': message, 'output': output})


class CCHandler(RpcHandler):
    """ Base class for handlers of CloudControl server.

    This class provide the following features:

    - functions can be used to get all functions decorated with the listed
        decorator
    - version can be used to get the current cc-server version
    """

    def __init__(self, client):
        self.client = client
        self.server = client.server  # Shortcut to server
        self.conf = client.server.conf  # Shortcut to configuration

    def __getitem__(self, name):
        if name.startswith('_'):
            # Filter the private members access:
            raise KeyError('Remote name %s is private.' % repr(name))
        else:
            self.logger.debug('Called %s.%s', self.__class__.__name__, name)
            return self._stats_decorator(super(CCHandler, self).__getitem__(name))

    def _stats_decorator(self, func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            call_id = base64.b64encode(os.urandom(6))
            started = time.time()
            returned = func(*args, **kwargs)
            duration = time.time() - started

            self.logger.info('Called handler [%s] by %s: %s(%s, %s) -> %.2f',
                             call_id, self.client.login, func.__name__, args,
                             kwargs, duration)

            return returned
        return wrapper

    @property
    def logger(self):
        return self.client.logger

    @listed
    def functions(self):
        """ Show the list of functions available to the peer.

        :return: list of dict with keys name and description.
        """

        cmd_list = []

        for attr in dir(self):
            attr = getattr(self, attr, None)
            if getattr(attr, '__listed__', False):
                cmd = {}
                cmd['name'] = attr.__name__
                doc = inspect.getdoc(attr)
                if doc:
                    cmd['description'] = inspect.cleandoc(doc)
                cmd_list.append(cmd)

        return cmd_list

    @listed
    def version(self):
        """ Return the current server version.

        :return: the version
        """
        return __version__
