# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


import re

from sjrpc.core import AsyncWatcher

from cloudcontrol.common.jobs import Job, JobCancelError

from cloudcontrol.server.utils import AcquiresAllOrNone
from cloudcontrol.server.exceptions import RightError


class CloneJob(Job):

    """ A clone job.

    Mandatory items:
     * vm_name: name of the vm to migrate
     * new_vm_name: the name of the cloned vm
     * hv_source: name of the hv which execute the VM
     * hv_dest: the destination hypervisor
     * author: login of the author cli
    """

    def job_type(self):
        return 'clone'

    def job(self, server, client, hv_source, vm_name, hv_dest, new_vm_name):
        self._func_cancel_xfer = None  # Callback to a function used to cancel
                                       # a disk transfert
        vm_id = '%s.%s' % (hv_source, vm_name)

        self.title = 'Clone %s --> %s' % (vm_id, hv_dest)
        self.logger.info('Started clone for %s', vm_id)

        # Cancel the job if the user has not the right to clone the vm or to
        # select an hypervisor as destination:
        try:
            client.check('clone', 'id=%s' % vm_id)
        except RightError:
            raise JobCancelError('author have no rights to clone this VM')

        try:
            client.check('clone', 'id=%s' % hv_dest)
        except RightError:
            raise JobCancelError('author have no right to clone to this hv')

        # Update the VM object:
        vm = server.db.get_by_id(vm_id)
        if vm is None:
            raise JobCancelError('Source VM not found')

        # Get the source and destination hv clients:
        try:
            source = server.get_client(hv_source)
        except KeyError:
            raise JobCancelError('source hypervisor is not connected')

        try:
            dest = server.get_client(hv_dest)
        except KeyError:
            raise JobCancelError('destination hypervisor is not connected')

        self.checkpoint()

        self.report('waiting lock for source and dest hypervisors')
        self.logger.info('Trying to acquire locks')

        with AcquiresAllOrNone(source.hvlock, dest.hvlock):
            self.logger.info('Locks acquired')
            self.checkpoint()

            before_clone_autostart = vm['autostart'].lower() == 'yes'

            # Create storages on destination:
            old_new_disk_mapping = {} # Mapping between old and new disk names
            names = {}
            self.report('create volumes')
            for disk in vm.get('disk', '').split():
                # Getting informations about the disk:
                pool = vm.get('disk%s_pool' % disk)
                name = vm.get('disk%s_vol' % disk)
                size = vm.get('disk%s_size' % disk)
                assert pool is not None, 'pool tag doesn\'t exists'
                assert name is not None, 'name tag doesn\'t exists'
                assert size is not None, 'size tag doesn\'t exists'

                # Change the name of the disk:
                old_name = name
                if name.startswith(vm_name):
                    suffix = name[len(vm_name):]
                    name = new_vm_name + suffix
                else:
                    name = '%s_%s' % (new_vm_name, name)

                names[disk] = name

                fulloldname = '/dev/%s/%s' % (pool, old_name)
                fullnewname = '/dev/%s/%s' % (pool, name)
                old_new_disk_mapping[fulloldname] = fullnewname

                # Create the volume on destination:
                dest.proxy.vol_create(pool, name, int(size))
                self.logger.info('Created volume %s/%s on destination '
                                 'hypervisor (was %s)', pool, name, old_name)

                # Rollback stuff for this action:
                def rb_volcreate():
                    dest.proxy.vol_delete(pool, name)
                self.checkpoint(rb_volcreate)

            # Define VM:
            self.report('define vm')
            self.logger.info('XML configuration transfert')
            vm_config = source.proxy.vm_export(vm_name)

            # Change vm configuration XML to update it with new name:
            new_vm_config = self._update_xml(vm_config, vm_name,
                                             new_vm_name,
                                             old_new_disk_mapping)

            dest.proxy.vm_define(new_vm_config)

            # Rollback stuff for vm definition:
            def rb_define():
                dest.proxy.vm_undefine(name)
            self.checkpoint(rb_define)

            # Copy all source disk on destination disk:
            for disk, name in names.iteritems():
                self._copy_disk(source, dest, vm, disk, name)

            # Setup autostart as it was before the migration
            dest.proxy.vm_set_autostart(new_vm_name,
                                        before_clone_autostart)

            self.logger.info('Cloning completed with success')


    def _update_xml(self, vm_config, old_name, name, old_new_name_mapping):
        """ Update the XML definition of the VM with the new vm name, the new
            vm disk, and remove the uuid tag.
        """

        vm_config = vm_config.replace('<name>%s</name>' % old_name,
                                      '<name>%s</name>' % name)
        vm_config = re.sub('<uuid>.*?</uuid>\n', '', vm_config)
        # delete MAC address, then it will be regenerated by libvirt
        vm_config = re.sub('<mac .*?/>\n', '', vm_config)
        for old, new in old_new_name_mapping.iteritems():
            vm_config = vm_config.replace("='%s'" % old,
                                          "='%s'" % new)
        return vm_config

    def _copy_disk(self, source, dest, vm, disk, new_disk):
        """ Copy the specified disk name of the vm from source to dest.
        """

        # Get informations about the disk:
        pool = vm.get('disk%s_pool' % disk)
        name = vm.get('disk%s_vol' % disk)
        self.report('copy %s/%s' % (pool, name))

        # Make the copy and wait for it end:
        xferprop = dest.proxy.vol_import(pool, new_disk)

        # Register the cancel function:
        def cancel_xfer():
            dest.proxy.vol_import_cancel(xferprop['id'])
        self._func_cancel_xfer = cancel_xfer

        # Wait for the end of transfert:
        watcher = AsyncWatcher()
        watcher.register(source.conn, 'vol_export', pool, name, dest.ip, xferprop['port'])
        watcher.register(dest.conn, 'vol_import_wait', xferprop['id'])

        msgs = watcher.wait()
        self._func_cancel_xfer = None

        # Compare checksum of two answers:
        checksums = []
        assert len(msgs) == 2

        for msg in msgs:
            if msg.get('error') is not None:
                msg = 'error while copy: %s' % msg['error']['message']
                raise JobCancelError(msg)
            else:
                checksums.append(msg['return'].get('checksum'))
                self.checkpoint()

        if checksums[0] != checksums[1]:
            raise JobCancelError('checksum mismatches')

    def cancel(self):
        if self._func_cancel_xfer is not None:
            self._func_cancel_xfer()
        super(CloneJob, self).cancel()
