# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


from sjrpc.core import AsyncWatcher

from cloudcontrol.common.jobs import Job, JobCancelError

from cloudcontrol.server.utils import AcquiresAllOrNone
from cloudcontrol.server.exceptions import RightError


class ColdMigrationJob(Job):

    """ A cold vm migration job.

    Mandatory items:
     * vm_name: name of the vm to migrate
     * hv_source: name of the hv which execute the VM
     * hv_dest: the destination hypervisor
     * author: login of the author cli
    """

    def job_type(self):
        return 'migration.cold'

    def job(self, server, client, hv_source, vm_name, hv_dest):
        self._func_cancel_xfer = None  # Callback to a function used to cancel
                                       # a disk transfert
        vm_id = '%s.%s' % (hv_source, vm_name)

        self.title = 'Cold migration %s --> %s' % (vm_id, hv_dest)
        self.logger.info('Started migration for %s', vm_id)

        # Cancel the job if the user has not the right to migrate the vm or to
        # select an hypervisor as destination:
        try:
            client.check('migrate', 'id=%s' % vm_id)
        except RightError:
            raise JobCancelError('author have no rights to migrate this VM')

        try:
            client.check('migrate', 'id=%s' % hv_dest)
        except RightError:
            raise JobCancelError('author have no right to migrate to this hv')

        # Update the VM object:
        vm = server.db.get_by_id(vm_id)
        if vm is None:
            raise JobCancelError('Source VM not found')

        # Get the source and destination hv clients:
        try:
            source = server.get_client(hv_source)
        except KeyError:
            raise JobCancelError('source hypervisor is not connected')

        try:
            dest = server.get_client(hv_dest)
        except KeyError:
            raise JobCancelError('destination hypervisor is not connected')

        self.checkpoint()

        self.report('waiting lock for source and dest hypervisors')

        with AcquiresAllOrNone(source.hvlock, dest.hvlock):
            self.logger.info('Locks acquired')
            self.checkpoint()

            if not vm['status'] == 'stopped':
                raise JobCancelError('vm is not stopped')

            before_migrate_autostart = vm['autostart'].lower() == 'yes'

            # Create storages on destination:
            self.report('create volumes')
            for disk in vm.get('disk', '').split():
                # Getting informations about the disk:
                pool = vm.get('disk%s_pool' % disk)
                name = vm.get('disk%s_vol' % disk)
                size = vm.get('disk%s_size' % disk)
                assert pool is not None, 'pool tag doesn\'t exists'
                assert name is not None, 'name tag doesn\'t exists'
                assert size is not None, 'size tag doesn\'t exists'

                # Create the volume on destination:
                dest.proxy.vol_create(pool, name, int(size))
                self.logger.info('Created volume %s/%s on destination '
                                 'hypervisor', pool, name)

                # Rollback stuff for this action:
                def rb_volcreate():
                    dest.proxy.vol_delete(pool, name)
                self.checkpoint(rb_volcreate)

            # Define VM:
            self.report('define vm')
            self.logger.info('XML configuration transfert')
            vm_config = source.proxy.vm_export(vm_name)
            dest.proxy.vm_define(vm_config)

            # Rollback stuff for vm definition:
            def rb_define():
                dest.proxy.vm_undefine(vm_name)
            self.checkpoint(rb_define)

            # Copy all source disk on destination disk:
            for disk in vm.get('disk', '').split():
                self._copy_disk(source, dest, vm, disk)

            # At this point, if operation is a success, all we need is just to
            # cleanup source hypervisor from disk and vm. This operation *CAN'T*
            # be cancelled or rollbacked if anything fails (unlikely). The
            # migration must be considered as a success, and the only way to
            # undo this is to start a new migration in the other way.

            # Delete the rollback list.
            # This is mandatory to avoid data loss if the cleanup
            # code below fail.
            self._wayback = []

            # Cleanup the disks:
            for disk in vm.get('disk', '').split():
                pool = vm.get('disk%s_pool' % disk)
                name = vm.get('disk%s_vol' % disk)

                source.proxy.vol_delete(pool, name)

            # Cleanup the VM:
            source.proxy.vm_undefine(vm_name)

            # Setup autostart as it was before the migration
            dest.proxy.vm_set_autostart(vm_name,
                                        before_migrate_autostart)

            self.logger.info('Migration completed with success')


    def _copy_disk(self, source, dest, vm, disk):
        """ Copy the specified disk name of the vm from source to dest.
        """

        # Get informations about the disk:
        pool = vm.get('disk%s_pool' % disk)
        name = vm.get('disk%s_vol' % disk)
        self.logger.info('Started copy for %s/%s', pool, name)
        self.report('copy %s/%s' % (pool, name))

        # Make the copy and wait for it end:
        xferprop = dest.proxy.vol_import(pool, name)

        # Register the cancel function:
        def cancel_xfer():
            dest.proxy.vol_import_cancel(xferprop['id'])
        self._func_cancel_xfer = cancel_xfer

        # Wait for the end of transfert:
        watcher = AsyncWatcher()
        watcher.register(source.conn, 'vol_export', pool, name, dest.ip, xferprop['port'])
        watcher.register(dest.conn, 'vol_import_wait', xferprop['id'])

        msgs = watcher.wait()
        self._func_cancel_xfer = None

        # Compare checksum of two answers:
        checksums = []
        assert len(msgs) == 2
        for msg in msgs:
            if msg.get('error') is not None:
                msg = 'error while copy: %s' % msg['error']['message']
                raise JobCancelError(msg)
            else:
                checksums.append(msg['return'].get('checksum'))
                self.checkpoint()

        if checksums[0] != checksums[1]:
            raise JobCancelError('checksum mismatches')

    def cancel(self):
        if self._func_cancel_xfer is not None:
            self._func_cancel_xfer()
        super(ColdMigrationJob, self).cancel()
