# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.

from sjrpc.core import AsyncWatcher

from cloudcontrol.common.jobs import Job, JobCancelError


class DiskCopyJob(Job):

    """ Copy a disk between two nodes.
    """

    def job_type(self):
        return 'diskcopy'

    def job(self, server, client, source_id, source_pool, source_vol, dest_id, dest_pool, dest_vol):
        self._func_cancel_xfer = None  # Callback to a function used to cancel a disk transfert

        env = (source_id, source_pool, source_vol, dest_id, dest_pool, dest_vol)
        self.title = 'Started disk copy between %s:%s/%s -> %s:%s/%s' % env

        source = server.get_client(source_id)
        dest = server.get_client(dest_id)

        xferprop = dest.proxy.vol_import(dest_pool, dest_vol)

        # Wait for the end of transfert:
        watcher = AsyncWatcher()
        watcher.register(source.conn, 'vol_export', source_pool, source_vol, dest.ip, xferprop['port'])
        watcher.register(dest.conn, 'vol_import_wait', xferprop['id'])

        msgs = watcher.wait()

        # Compare checksum of two answers:
        checksums = []
        assert len(msgs) == 2
        for msg in msgs:
            if msg.get('error') is not None:
                msg = 'error while copy: %s' % msg['error']['message']
                raise JobCancelError(msg)
            else:
                checksums.append(msg['return'].get('checksum'))
                self.checkpoint()

        if checksums[0] != checksums[1]:
            raise JobCancelError('checksum mismatches')
