# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


import time

from sjrpc.core import AsyncWatcher

from cloudcontrol.common.jobs import Job, JobCancelError

from cloudcontrol.server.utils import AcquiresAllOrNone
from cloudcontrol.server.exceptions import RightError


class HotMigrationJob(Job):

    """ A hot vm migration job.

    Mandatory items:
     * vm_name: name of the vm to migrate
     * hv_source: name of the hv which execute the VM
     * hv_dest: the destination hypervisor
     * author: login of the author cli
    """

    def job_type(self):
        return 'migration.hot'

    def job(self, server, client, hv_source, vm_name, hv_dest):
        vm_id = '%s.%s' % (hv_source, vm_name)

        self.title = 'Hot migration %s --> %s' % (vm_id, hv_dest)
        self.logger.info('Started hot migration for %s', vm_id)

        # Cancel the job if the user has not the right to migrate the vm or to
        # select an hypervisor as destination:
        try:
            client.check('migrate', 'id=%s' % vm_id)
        except RightError:
            raise JobCancelError('author have no rights to migrate this VM')

        try:
            client.check('migrate', 'id=%s' % hv_dest)
        except RightError:
            raise JobCancelError('author have no right to migrate to this hv')

        # Update the VM object:
        vm = server.db.get_by_id(vm_id)
        if vm is None:
            raise JobCancelError('Source VM not found')

        # Get the source and destination hv clients:
        try:
            source = server.get_client(hv_source)
        except KeyError:
            raise JobCancelError('source hypervisor is not connected')

        try:
            dest = server.get_client(hv_dest)
        except KeyError:
            raise JobCancelError('destination hypervisor is not connected')

        self.checkpoint()

        self.report('waiting lock for source and dest hypervisors')

        with AcquiresAllOrNone(source.hvlock, dest.hvlock):
            self.logger.info('Locks acquired')
            self.checkpoint()

            before_clone_autostart = vm['autostart'].lower() == 'yes'


            if not vm['status'] == 'running':
                raise JobCancelError('vm is not stopped')

            to_cleanup = []

            # Create storages on destination and start synchronization:
            disks = vm.get('disk', '').split()
            for disk in disks:
                to_cleanup += self._sync_disk(vm, disk, source, dest)

            # Libvirt tunnel setup:
            tunres_src = source.proxy.tun_setup()
            def rb_tun_src():
                source.proxy.tun_destroy(tunres_src)
            self.checkpoint(rb_tun_src)

            tunres_dst = dest.proxy.tun_setup(local=False)
            def rb_tun_dst():
                dest.proxy.tun_destroy(tunres_dst)
            self.checkpoint(rb_tun_dst)

            source.proxy.tun_connect(tunres_src, tunres_dst, dest.ip)
            dest.proxy.tun_connect_hv(tunres_dst, migration=False)

            # Migration tunnel setup:
            migtunres_src = source.proxy.tun_setup()
            def rb_migtun_src():
                source.proxy.tun_destroy(migtunres_src)
            self.checkpoint(rb_migtun_src)

            migtunres_dst = dest.proxy.tun_setup(local=False)
            def rb_migtun_dst():
                dest.proxy.tun_destroy(migtunres_dst)
            self.checkpoint(rb_migtun_dst)

            source.proxy.tun_connect(migtunres_src, migtunres_dst, dest.ip)
            dest.proxy.tun_connect_hv(migtunres_dst, migration=True)

            # Initiate the live migration:
            self.report('migration in progress')
            source.proxy.vm_migrate_tunneled(vm_name, tunres_src,
                                             migtunres_src, _timeout=None)

            # At this point, if operation is a success, all we need is just to
            # cleanup source hypervisor from disk and vm. This operation *CAN'T*
            # be cancelled or rollbacked if anything fails (unlikely). The
            # migration must be considered as a success, and the only way to
            # undo this is to start a new migration in the other way.

            # Delete the rollback list.
            # This is mandatory to avoid data loss if the cleanup
            # code below fail.
            self.report('cleanup')
            self._wayback = []

            source.proxy.tun_destroy(tunres_src)
            dest.proxy.tun_destroy(tunres_dst)
            source.proxy.tun_destroy(migtunres_src)
            dest.proxy.tun_destroy(migtunres_dst)

            for cb_cleanup in reversed(to_cleanup):
                cb_cleanup()

            # Cleanup the disks:
            for disk in disks:
                pool = vm.get('disk%s_pool' % disk)
                name = vm.get('disk%s_vol' % disk)

                source.proxy.vol_delete(pool, name)

            # Setup autostart as it was before the migration
            dest.proxy.vm_set_autostart(vm_name,
                                        before_clone_autostart)

            self.logger.info('Migration completed with success')


    def _sync_disk(self, vm, disk, source, dest):
        to_cleanup = []

        # getting informations about the disk:
        pool = vm.get('disk%s_pool' % disk)
        name = vm.get('disk%s_vol' % disk)
        size = vm.get('disk%s_size' % disk)
        assert pool is not None, 'pool tag doesn\'t exists'
        assert name is not None, 'name tag doesn\'t exists'
        assert size is not None, 'size tag doesn\'t exists'

        status_msg = 'sync volume %s/%s (%%s)' % (pool, name)
        self.report(status_msg % 'creation')

        # create the volume on destination:
        dest.proxy.vol_create(pool, name, int(size))
        self.logger.info('Created volume %s/%s on destination '
                         'hypervisor', pool, name)

        # rollback stuff for this action:
        def rb_volcreate():
            dest.proxy.vol_delete(pool, name)
        self.checkpoint(rb_volcreate)

        # setup the drbd synchronization with each hypervisors:
        self.report(status_msg % 'setup')

        res_src = source.proxy.drbd_setup(pool, name)
        def rb_setupsrc():
            source.proxy.drbd_shutdown(res_src)
        self.checkpoint(rb_setupsrc)
        to_cleanup.append(rb_setupsrc)

        res_dst = dest.proxy.drbd_setup(pool, name)
        def rb_setupdst():
            dest.proxy.drbd_shutdown(res_dst)
        self.checkpoint(rb_setupdst)
        to_cleanup.append(rb_setupdst)

        # start connection of drbd:
        self.report(status_msg % 'connect')

        watcher = AsyncWatcher()
        watcher.register(source.conn, 'drbd_connect', res_src, res_dst, dest.ip)
        watcher.register(dest.conn, 'drbd_connect', res_dst, res_src, source.ip)
        msgs = watcher.wait(timeout=30)
        for msg in msgs:
            if 'error' in msg:
                msg = 'error while drbd_connect: %s' % msg['error']['message']
                raise JobCancelError(msg)

        # setup topology as primary/secondary:
        source.proxy.drbd_role(res_src, True)
        dest.proxy.drbd_role(res_dst, False)

        # Wait the end of the synchronization:
        sync_running = True
        while sync_running:
            status = dest.proxy.drbd_sync_status(res_dst)
            if status['done']:
                sync_running = False

            self.report(status_msg % 'sync %s%%' % status['completion'])
            time.sleep(2)

        dest.proxy.drbd_role(res_dst, True)

        source.proxy.drbd_takeover(res_src, True)
        def rb_takeover_src():
            source.proxy.drbd_takeover(res_src, False)
        self.checkpoint(rb_takeover_src)
        to_cleanup.append(rb_takeover_src)

        dest.proxy.drbd_takeover(res_dst, True)
        def rb_takeover_dst():
            dest.proxy.drbd_takeover(res_dst, False)
        self.checkpoint(rb_takeover_dst)
        to_cleanup.append(rb_takeover_dst)

        return to_cleanup
