# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


import time

from cloudcontrol.common.jobs import Job

class KillOldCliJob(Job):

    """ Typically an hidden job used to kill clients who are connected/idle
        since too much time.

    Mandatory items:
     * maxcon: maximum connection time in minutes
     * maxidle: maximum idle time in minutes

    Optional items:
     * delay: delay in secondes between two checks (default 1m)
    """

    DEFAULT_DELAY = 60

    def job(self, server, maxcon, maxidle, delay=DEFAULT_DELAY):
        self.state.title = 'Kill old clients (delay=%s)' % delay
        while True:
            self.checkpoint()
            for client in list(server.iterclients('cli')):
                if client.uptime > (maxcon * 60) or client.idle > (maxidle * 60):
                    server.kill(client.login)
                    self.logger.info('Disconnected %s because of its long time'
                                     ' or idle connection', client.login)
            time.sleep(delay)
