# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


""" An interface to the server for the JobsManager
"""

from datetime import datetime

from cloudcontrol.server.db import SObject

from cloudcontrol.common.jobs import JobsManagerInterface
from cloudcontrol.common.tql.db.tag import StaticTag, CallbackTag
from cloudcontrol.common.tql.db.helpers import taggify


class ServerJobsManagerInterface(JobsManagerInterface):

    TAG_ATTRIBUTES = ('title', 'status', 'state', 'owner', 'created', 'ended',
                      'attachments', 'batch', 'jid', 'type')

    def __init__(self, server):
        self._server = server
        self._jobs = {}  # Store each TQL object relative to a job

    def on_job_created(self, job):
        tql_object = SObject(job.id)
        tql_object.register(StaticTag('r', 'job'))
        self._server.db.register(tql_object)
        self._jobs[job.id] = tql_object

    def on_job_updated(self, job):
        tql_object = self._jobs[job.id]

        for name in self.TAG_ATTRIBUTES:
            value = getattr(job, name, None)
            if value is None:
                continue
            value = taggify(value)
            if name not in tql_object:
                tag = StaticTag(name, '')
                tql_object.register(tag)
            else:
                tag = tql_object[name]
            tag.value = value

        if 'duration' not in tql_object:
            def compute_duration():
                end = datetime.now() if job.ended is None else job.ended
                dt = end - job.created
                return dt.seconds + dt.days * 86400
            tql_object.register(CallbackTag('duration', compute_duration, ttl=0))

    def on_job_purged(self, job_id):
        tql_object = self._jobs[job_id]
        self._server.db.unregister(tql_object)
        del self._jobs[job_id]
