# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


""" Riskgroup manager.
"""

import json


class RiskgroupsManager(object):

    """ Handle persistent storage of riskgroups.
    """

    def __init__(self, logger, riskgroup_filename):
        self.logger = logger
        self._riskgroup_filename = riskgroup_filename
        self._groups = {}
        self.load()

    def load(self):
        """ Load groups.
        """
        try:
            with open(self._riskgroup_filename) as fgroups:
                self._groups = json.load(fgroups)
        except IOError as err:
            if err.errno == 2:  # No such file, loading empty group
                self._groups = {}
            else:
                raise  # Reraise any other error
        self.logger.info('Loaded %d riskgroups', len(self._groups))

    def save(self):
        """ Save the current riskgroups to disk.
        """
        with open(self._riskgroup_filename, 'w') as fgroups:
            json.dump(self._groups, fgroups)

    @property
    def groups(self):
        return self._groups

    def get(self, name, default=None):
        """ Get the specified group or default.
        """

        return self._groups.get(name, default)

    def set(self, name, properties):
        """ Set a new or already existing group.
        """

        self._groups[name] = properties
        self.save()

    def delete(self, name):
        """ Delete the specified group.
        """

        del self._groups[name]
        self.save()
