# This file is part of CloudControl.
#
# CloudControl is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# CloudControl is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with CloudControl.  If not, see <http://www.gnu.org/licenses/>.


""" Some helpers used by cc-server.
"""

from threading import Lock

class Acquires(object):

    """ Context manager used to acquire more than one lock at once. It works
        rely on the fact that if locks are always acquired in the same order,
        we can't enter in a deadlock situation.

    Usage is very simple:

    >>> a = Lock()
    >>> b = Lock()
    >>> with Acquires(b, a):
    ...     print 'locked'
    ...

    .. seealso::
        http://dabeaz.blogspot.com/2009/11/python-thread-deadlock-avoidance_20.html
    """

    def __init__(self, *locks):
        self._locks = sorted(set(locks), key=lambda x: id(x))

    def __enter__(self):
        for lock in self._locks:
            lock.acquire()

    def __exit__(self, exc_type, exc_value, traceback):
        for lock in self._locks:
            lock.release()


class AcquiresAllOrNone(Acquires):

    """ Class that extend Acquires to allow to release all lock if one of
        them is not free.
    """

    # Global acquire lock:
    acquirelock = Lock()

    def __enter__(self):
        while True:
            with self.acquirelock:
                acquired = []
                for lock in self._locks:
                    if not lock.acquire(False):
                        for lock_acquired in acquired:
                            lock_acquired.release()
                        break
                    else:
                        acquired.append(lock)
                else:
                    break


def itercounter(iterator, callback, *args, **kwargs):

    """ Iterate over an iterator and call callback with number of iterations.
    """

    count = 0
    for item in iterator:
        count += 1
        yield item
    callback(count, *args, **kwargs)
