#!/usr/bin/env python
#coding=utf8

'''
CloudControl CLI main module
'''

import os, os.path
import sys
import re
import subprocess

from cccli import version, debug
from cccli.command import Command, Alias
from cccli.printer import Printer
from cccli.exception import *

from sjrpc.client import SimpleRpcClient
from sjrpc.utils import RpcHandler, ConnectionProxy, pure
from sjrpc.core.exceptions import *

class Cli(object):
    def __init__(self, settings):
        self.isinteractive = sys.stderr.isatty() and sys.stdin.isatty()
        self.settings = settings
        self.prompt = "> "
        self.rpc = None
        self.printer = None
        self.alias = Alias()

    def start(self, line=""):
        '''Start a CLI'''
        # not interactive if command line
        if line:
            self.isinteractive = False
        # start printer and load history
        self.printer = Printer(self.isinteractive)
        self.printer.history.load(self.settings.get("history", ""))
        self.printer.history.maxsize(self.settings.get("hsize", None))
        # load alias
        self.alias.load(self.settings.get("alias", ""))
        self.printer.debug("Alias: %s"%self.alias)
        # Connecting
        self._connect()
        # authentifications
        self._auth()
        # run parsing args
        if line:
            self._parse_line(line)
        else:
            self._shell()
        self.printer.history.save(self.settings.get("history", ""))


    def _connect(self):
        self.printer.debug("Connecting...")
        rpcc = SimpleRpcClient.from_addr(self.settings["server"],
                                         self.settings["port"],
                                         enable_ssl=True,
                                         default_handler=CliHandler(),
                                         on_disconnect="quit",
                                         timeout=self.settings["timeout"]
                                        )
        rpcc.start(daemonize=True)
        self.rpc = ConnectionProxy(rpcc)

    def _auth(self):
        self.printer.debug("Authenticating...")
        if self.rpc.authentify(self.settings["login"], self.settings["pass"]):
            self.printer.debug("Authenticated.")
        else:
            self.printer.fatal("Autentification failed!")

    def _shell(self):
        '''Shell parser'''
        while True:
            try:
                line = self.printer.getline(self.prompt)
                self._parse_line(line)
            except EOFError:
                self.printer.out("")
                break
            except SystemExit:
                break
            except KeyboardInterrupt:
                self.printer.out("")
        self.printer.out("Tcho!")

    def _parse_line(self, line):
        '''Parse a line (more than one command)'''
        for cmd in line.split(";"):
            if (cmd.strip() == "" or cmd[0] == "#"):
                continue
            elif (cmd[0] == "!"):
                p = subprocess.Popen(cmd[1:], close_fds=True, shell=True)
                p.wait()
                ret = p.returncode
            elif (cmd[0] == "?"):
                Command(["help"], self).call()
            else:
                self._parse_command(cmd)

    def _parse_command(self, cmd):
        try:
            # lex command
            argv = self._lex_argv(cmd)
            # alias subs
            if argv[0] in self.alias:
                argv[0] = self.alias[argv[0]]
            # execute command
            Command(argv, self).call()
        except BadArgument, e:
            if str(e):
                self.printer.error("Bad argument: %s."%str(e))
            else:
                self.printer.error("Bad argument.")
            usage = Command.usage(argv[0])
            if usage != "":
                self.printer.out("usage: %s."%usage)
        except BadCommand, e:
            if str(e):
                self.printer.error("command: %s."%str(e))
            else:
                self.printer.error("No command: %s."%argv[0])
        except RpcError, e:
            if debug: raise
            self.printer.error("sjRPC: %s"%str(e))
        except Exception, e:
            if debug: raise
            self.printer.error("%s: %s."%(argv[0], str(e)))

    def _lex_argv(self, string):
        '''Lex command argument'''
        return string.split(" ")

class CliHandler(RpcHandler):
    '''Handle RPC incoming request'''

    @pure
    def get_tags(self, tags=()):
        if "version" in tags:
            return { "version": version }

    @pure
    def quit(self, rpc=None):
        self.printer.fatal("Disconnected from server!")
