#!/usr/bin/env python
#coding=utf8

'''
CloudControl CLI command module
'''

from cccli.exception import *
from sjrpc.core.exceptions import *
from cccli.printer import Printer, color
from optparse import OptionParser

class Command(object):
    '''Base of all command class'''

    def __init__(self, cli, argv0):
        self.cli = cli
        self.printer = self.cli.printer
        self.name = argv0

    def __call__(self, argv):
        raise NotImplementedError

    def usage(self):
        return "Usage: %s"%self.name

    def help(self):
        return self.__doc__


class OptionCommand(Command):
    '''Add options parser to Command'''

    class OptionCommandParser(OptionParser):
        '''Parser of Option for OptionCommand'''

        def error(self, e):
            raise cmdBadArgument(e)

        def exit(self):
            raise cmdExit()

    def __init__(self, cli, argv0):
        Command.__init__(self, cli, argv0)
        self.optionparser = OptionCommand.OptionCommandParser(prog=argv0)
        self.set_usage("%prog [options]")
        self.options = None
        self.args = list()

    def usage(self):
        '''Return usage string'''
        return self.optionparser.format_help().strip()

    def parse_args(self, argv):
        '''Wrapper to parse_args'''
        (self.options, self.args) = self.optionparser.parse_args(argv[1:])

    def add_option(self, *args, **kwargs):
        '''Proxy to OptionParser'''
        self.optionparser.add_option(*args, **kwargs)

    def remove_option(self, *args, **kwargs):
        '''Proxy to OptionParser'''
        self.optionparser.remove_option(*args, **kwargs)

    def set_usage(self, *args, **kwargs):
        '''Proxy to OptionParser'''
        self.optionparser.set_usage(*args, **kwargs)

class TqlCommand(OptionCommand):
    '''Add Tql stuff to Command'''

    def __init__(self, cli, argv0):
        OptionCommand.__init__(self, cli, argv0)
        self.rpc = cli.rpc
        self.set_usage("%prog [options] <tql>")
        # set tql status stuff
        self.add_option("-s", "--status", action="store_true", dest="status",
                        help="Show status of each tql object")
        # set tagdisplay stuff
        self.tdr = self.cli.tagdisplay.resolve
        self.tdc = self.cli.tagdisplay.color
        self.tdtc = self.cli.tagdisplay.titlecolor
        self.add_option("-n", "--no-tagdisplay", action="callback", dest="tagdisplay",
                        callback=self._cb_notagdisplay,
                        help="No tagdisplay custom display")


    def _cb_notagdisplay(self, option, opt, value, parser):
        '''Callback for option --no-tagdisplay'''
        self.tdr = lambda tagname, tagvalue: tagvalue
        self.tdc = self.cli.tagdisplay.default_color
        self.tdtc = self.cli.tagdisplay.default_titlecolor

    def rpccall(self, *args, **kwargs):
        '''Call a RPC method an show tql return'''
        try:
            d = self.rpc.call(*args, **kwargs)
            if self.options.status:
                self.show_status(d)
            return d
        except RpcError as e:
            raise cmdError("RPCError: %s"%str(e))


    def show_status(self, ans):
        '''Show status of an Tql request'''
        try:
            self.printer.out("Status:")
            for o in ans:
                s = "%sid: %s%s %sstatus: %s%s %smessage:%s%s%s"%(
                    self.tdtc("id"), self.tdc("id"), self.tdr("id", o),
                    self.tdtc("status"), self.tdc("status"), ans[o][0],
                    self.tdtc("message"), self.tdc("message"), ans[o][1],
                    color["reset"])
                if s:
                    self.printer.out(s)
        except Exception:
            pass
