#!/usr/bin/env python
#coding=utf8

'''
CloudControl list command
'''

from cccli.exception import *
from sjrpc.core.exceptions import *
from cccli.printer import Printer, color
from cccli.command.command import OptionCommand

class Command_list(OptionCommand):
    '''List objects'''

    def __init__(self, cli, argv0):
        OptionCommand.__init__(self, cli, argv0)
        self.set_usage("%prog [options] [tql]")
        self.add_option("-t", action="store_true", dest="table",
                        help="column aligment display")
        self.add_option("-l", action="store_true", dest="align",
                        help="line aligment display")
        self.add_option("-n", "--no-tagdisplay", action="store_false", dest="tagdisplay", default=True,
                        help="No tag display system")

    def __call__(self, argv):
        self.parse_args(argv)
        if self.options.tagdisplay:
            self.td = self.cli.tagdisplay.resolve
            self.tc = self.cli.tagdisplay.color
        else:
            self.td = lambda tagname, tagvalue: unicode(tagvalue)
            self.tc = lambda tagname: color["reset"]
        if len(self.args) == 0:
            self.args.append("")
        try:
            objs = self.cli.rpc.call("list", str.join("", self.args))
        except RpcError as e:
            raise cmdError("RPCError: %s"%str(e))
        if len(objs) == 0:
            return
        if self.options.align:
            self._list_align(objs)
        elif self.options.table:
            self._list_table(objs)
        else:
            self._trivial_list(objs)

    def _trivial_list(self, objs):
        '''Trivial listing of tag'''
        for o in objs:
            id = self.td("id", o.pop("id"))
            tags = " ".join([ "%s%s:%s%s"%(color["reset"],
                                           t,
                                           self.tc(t),
                                           self.td(t, v))
                              for (t,v) in o.items() ])
            self.printer.out("%sid:%s%s %s%s"%(color["reset"], self.tc("id"), id, tags, color["reset"]))

    def _list_align(self, objs):
        '''Listing line aligned'''
        # get max size by tag
        tags = dict()
        for o in objs:
            for t,v in o.items():
                tags[t] = max(len(self.td(t, v)), tags.get(t, len(str(t))))
        # extract id size
        idsize = tags.pop("id")
        # dislay each object by line
        for o in objs:
            # show id tag
            line = "%sid:%s%s"%(color["reset"],
                                  self.tc("id"),
                                  self.td("id", o.pop("id")).ljust(idsize + 2))
            # show others tags
            for tagname in sorted(tags.keys()):
                line += "%s%s:%s%s"%(color["reset"],
                                      tagname,
                                      self.tc(tagname),
                                      self.td(tagname, o.get(tagname, "")).ljust(tags[tagname] + 1))
            self.printer.out("%s%s"%(line, color["reset"]))

    def _list_table(self, objs):
        '''Listing table style'''
        # get max size by tag
        tags = dict()
        for o in objs:
            for t,v in o.items():
                tags[t] = max(len(self.td(t, v)), tags.get(t, len(str(t))))
        # extract id size
        idsize = tags.pop("id")
        # print id title
        self.printer.out(color["reset"], nl="")
        self.printer.out("id".ljust(idsize+1), nl=" ")
        # print others titles
        for t,v in tags.items():
            self.printer.out(t.ljust(v), nl=" ")
        self.printer.out(color["reset"])
        # print obj
        for obj in objs:
            # print id first
            self.printer.out(self.tc("id"), nl="")
            self.printer.out(self.td("id", obj.pop("id")).ljust(idsize+1), nl=" ")
            # print others tags
            for (t, v) in tags.items():
                self.printer.out(self.tc(t), nl="")
                self.printer.out(self.td(t, obj.get(t, "")).ljust(v) ,nl=" ")
            self.printer.out(color["reset"])
