#!/usr/bin/env python
#coding=utf8

'''
CloudControl Tag displaying stuff
'''

from cccli.exception import *
from cccli.printer import Printer, color
import os
import re
import math
import ConfigParser
import time
import fnmatch
import datetime

class TagDisplay(object):
    '''Handle tagdisplay stuff'''

    def __init__(self):
        self.option = dict()
        self.tagtype = dict()
        self.tagcolor = dict()
        self.tagtitlecolor = dict()
        self.types = [ x[5:] for x in dir(self) if x.startswith("type_") ]

    def load(self, filename):
        '''load tagdisplay settings from file'''
        if filename is not None:
            fparser = ConfigParser.RawConfigParser()
            fparser.read(filename)
            self.__init__()
            if fparser.has_section("option"):
                self.option.update(fparser.items("option"))
            if fparser.has_section("type"):
                self.tagtype.update(fparser.items("type"))
            if fparser.has_section("color"):
                self.tagcolor.update(fparser.items("color"))
            if fparser.has_section("titlecolor"):
                self.tagtitlecolor.update(fparser.items("titlecolor"))

    def save(self, filename):
        '''save tagdisplay settings on file'''
        if filename is not None:
            fparser = ConfigParser.RawConfigParser()
            fparser.read(filename)
            for n,d in (("type", self.tagtype), ("color", self.tagcolor),
                        ("option", self.option), ("titlecolor", self.tagtitlecolor)):
                fparser.remove_section(n)
                fparser.add_section(n)
                for k,v in d.items():
                    fparser.set(n, k, v)
            fparser.write(open(filename, "w"))

    def color(self, tagname):
        '''Return the current tag color'''
        # build list of matching pattern with tagname
        l = [ x for x in self.tagcolor if fnmatch.fnmatch(tagname, x) ]
        if len(l) > 0:
            # select longest match
            tm = max(l)
            if self.tagcolor[tm] in color:
                return color[self.tagcolor[tm]]
        return self.default_color(tagname)

    def default_color(self, tagname):
        '''Return default color'''
        return color["yellow"] if tagname != "id" else color["lblue"]

    def titlecolor(self, tagname):
        # build list of matching pattern with tagname
        l = [ x for x in self.tagtitlecolor if fnmatch.fnmatch(tagname, x) ]
        if len(l) > 0:
            # select longest match
            tm = max(l)
            if self.tagtitlecolor[tm] in color:
                return color[self.tagtitlecolor[tm]]
        return self.default_titlecolor(tagname)

    def default_titlecolor(self, tagname):
        '''Return default title color'''
        return color["reset"]

    def resolve(self, tagname, tagvalue):
        '''Transform a tagvalue respecting custom display settings'''
        # check general options
        if bool(self.option.get("quotespace", False)):
            if re.search("\s", tagvalue) is not None:
                tagvalue = u"'%s'"%re.sub("'", "\'", tagvalue)
        # build list of matching pattern with tagname
        l = [ x for x in self.tagtype if fnmatch.fnmatch(tagname, x) ]
        if len(l) > 0:
            # select longest match
            tm = max(l)
            if self.tagtype[tm] in self.types:
                return getattr(self, "type_%s"%self.tagtype[tm])(tagvalue)
        return tagvalue

    def type_string(self, value):
        '''DO Nothing'''
        return value()

    def type_lower(self, value):
        '''Lower case type'''
        return value.lower()

    def type_upper(self, value):
        '''Upper case type'''
        return value.upper()

    def type_si(self, value):
        '''System International type'''
        if value.isdecimal():
            v = float(value)
            if v >= 1000:
                si = "KMGTPEZY"
                p = min(math.floor(math.log10(abs(v))/3.0), len(si))
                d = v / pow(10, 3*p)
                u = si[int(p-1)]
                value =  u"%.4g%s"%(d, u)
        return value

    def type_bit(self, value):
        '''Bit type'''
        if value.isdecimal():
            v = float(value)
            if v >= 1000:
                si = "KMGTPEZY"
                p = min(math.floor(math.log(abs(v), 2)/10.0), pow(2, len(si)))
                d = v / pow(2, 10*p)
                u = si[int(p-1)]
                value =  u"%.4g%si"%(d, u)
        return value

    def type_second(self, value):
        '''Second type'''
        if value.isdecimal():
            v = long(value)
            if v < 60:
                return u"%ss"%value
            elif v < 3600:
                return u"%dm%ds"%(v/60, v%60)
            elif v < 86400:
                return u"%dh%dm%ds"%(v/3600, v/60%60, v%60)
            else:
                return u"%dd%dh%dm%ds"%(v/86400, v/3600%24, v/60%60, v%60)
        return value

    def type_date(self, value):
        '''date type'''
        if value.isdecimal():
            d = datetime.datetime.fromtimestamp(float(value))
            return d.strftime("%d/%m/%Y")
        return value

    def type_time(self, value):
        '''date type'''
        if value.isdecimal():
            d = datetime.datetime.fromtimestamp(float(value))
            return d.strftime("%H:%M:%S")
        return value

    def type_datetime(self, value):
        '''date type'''
        if value.isdecimal():
            d = datetime.datetime.fromtimestamp(float(value))
            return d.strftime("%d/%m/%Y %H:%M:%S")
        return value
