
'''
CloudControl VM related commands
'''

from cccli.exception import *
from sjrpc.core.exceptions import *
from cccli.printer import Printer, color
from cccli.command.command import OptionCommand

class VmCommand(OptionCommand):
    '''Command for vm style'''

    def __init__(self, cli, argv0):
        OptionCommand.__init__(self, cli, argv0)
        self.option.set_usage("%prog [options] <tql>")
        self.option.add_option("--raw", action="store_true", dest="raw",
                               help="Don't append filter on request")
        self.option.add_option("--direct", action="store_true", dest="direct",
                               help="Directly send tql to server (don't list before)")
        self.option.add_option("--tg", action="store_true", dest="noask",
                               help="Don't ask confirmation (Dangerous)")

    def _vm_action(self, argv, filters=None):
        try:
            (options, args) = self.option.parse_args(argv[1:])
        except SystemExit:
            return
        if len(args) == 0:
            raise cmdBadArgument()
        tql = str.join("", args)
        # append securty options by command name
        if filters is not None and not options.raw:
            tql += filters
        if options.direct:
            try:
                objs = self.cli.rpc.call(argv[0], tql)
            except RpcError as e:
                raise cmdError("RPCError: %s"%str(e))
        else:
            # get objects id
            try:
                objs = self.cli.rpc.call("list", tql)
            except RpcError as e:
                raise cmdError("RPCError: %s"%str(e))
            # no result, goodbye
            if len(objs) == 0:
                raise cmdWarning("tql: '%s': No result."%tql)
            self.printer.out("You will %s:"%argv[0])
            for obj in objs:
                self.printer.out("%sid:%s%s%s"%(color["green"],color["yellow"],obj["id"],color["reset"]))
            self.printer.out("%sCount: %s%s"%(color["green"],color["reset"], len(objs)))
            # be sure boby want do that
            if not options.noask:
                self.printer.out("%sProceed?%s"%(color["lred"], color["reset"]))
                if self.printer.ask("Answer (yes/NO): ") != "yes":
                    raise cmdWarning("Aborted")
                if len(objs) > 5:
                    self.printer.out("%sYou request is on more than 5 objets!%s"%(color["yellow"], color["reset"]))
                    self.printer.out("%sAre you really sure?%s"%(color["lred"], color["reset"]))
                    if self.printer.ask("Answer (Sir, yes Sir!): ") != "Sir, yes Sir!":
                        raise cmdWarning("Bad Answer. Aborted")
            # execute action for each object
            for obj in objs:
                self.printer.out("%s%s%s %s%s%s"%(color["lblue"],
                                                  argv[0],
                                                  color["reset"],
                                                  color["yellow"],
                                                  obj["id"],
                                                  color["reset"]))
                try:
                    self.cli.rpc.call(argv[0], "id:%s"%obj["id"])
                except RpcError as e:
                    self.printer.error("RPCError: %s"%str(e))


class Command_start(VmCommand):
    '''Start a stopped vm'''
    def __call__(self, argv):
        self._vm_action(argv, "&r=vm&status=stopped")


class Command_stop(VmCommand):
    '''Stop a running vm'''

    def __call__(self, argv):
        self._vm_action(argv, "&r=vm&status=running")


class Command_destroy(VmCommand):
    '''Force a vm to stop'''

    def __call__(self, argv):
        self._vm_action(argv, "&r=vm&status!=stopped")

class Command_restart(VmCommand):
    '''Restart a vm'''

    def __call__(self, argv):
        self._vm_action(argv, "&r=vm")


class Command_pause(VmCommand):
    '''Pause a running vm'''

    def __call__(self, argv):
        self._vm_action(argv, "&r=vm&status=running")


class Command_resume(VmCommand):
    '''Resume a paused vm'''

    def __call__(self, argv):
        self._vm_action(argv, "&r=vm&status=paused")
